<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Supplier;
use Illuminate\Http\Request;

class PaymentToSupplierController extends Controller
{
    public function index()
    {
        $payments = Payment::where('type', 'to_supplier')->with('supplier')->latest()->get();
        return view('payments-to-supplier.index', compact('payments'));
    }

    public function create()
    {
        $suppliers = Supplier::all();
        return view('payments-to-supplier.create', compact('suppliers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'description' => 'nullable|string',
        ]);

        // Save the payment
        $payment = Payment::create([
            'supplier_id' => $request->supplier_id,
            'type' => 'to_supplier',
            'amount' => $request->amount,
            'source' => $request->source,
            'description' => $request->description,
        ]);

        // Update supplier debt balance
        $supplier = $payment->supplier;
        $toSupplierTotal = Payment::where('supplier_id', $supplier->id)
                                 ->where('type', 'to_supplier')
                                 ->sum('amount');
        $productPaymentTotal = Payment::where('supplier_id', $supplier->id)
                                     ->where('type', 'product_payment')
                                     ->sum('amount');
        $supplier->updateDebt($toSupplierTotal, $productPaymentTotal);

        return redirect()->route('payments-to-supplier.index')
                        ->with('success', 'Yetkazib beruvchiga to‘lov qo‘shildi va qarz balansi yangilandi');
    }

    public function edit(Payment $payment)
    {
        if ($payment->type !== 'to_supplier') {
            return redirect()->route('payments-to-supplier.index')->with('error', 'Bu to‘lov yetkazib beruvchiga emas');
        }
        $suppliers = Supplier::all();
        return view('payments-to-supplier.edit', compact('payment', 'suppliers'));
    }

    public function update(Request $request, Payment $payment)
    {
        if ($payment->type !== 'to_supplier') {
            return redirect()->route('payments-to-supplier.index')->with('error', 'Bu to‘lov yetkazib beruvchiga emas');
        }

        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'amount' => 'required|numeric|min:0',
            'source' => 'required|in:cash,card',
            'description' => 'nullable|string',
        ]);

        $payment->update([
            'supplier_id' => $request->supplier_id,
            'amount' => $request->amount,
            'source' => $request->source,
            'description' => $request->description,
        ]);

        // Update supplier debt balance
        $supplier = $payment->supplier;
        $toSupplierTotal = Payment::where('supplier_id', $supplier->id)
                                 ->where('type', 'to_supplier')
                                 ->sum('amount');
        $productPaymentTotal = Payment::where('supplier_id', $supplier->id)
                                     ->where('type', 'product_payment')
                                     ->sum('amount');
        $supplier->updateDebt($toSupplierTotal, $productPaymentTotal);

        return redirect()->route('payments-to-supplier.index')
                        ->with('success', 'To‘lov yangilandi va qarz balansi yangilandi');
    }

    public function destroy(Payment $payment)
    {
        if ($payment->type !== 'to_supplier') {
            return redirect()->route('payments-to-supplier.index')->with('error', 'Bu to‘lov yetkazib beruvchiga emas');
        }

        $supplier = $payment->supplier;
        $payment->delete();

        // Update supplier debt balance
        $toSupplierTotal = Payment::where('supplier_id', $supplier->id)
                                 ->where('type', 'to_supplier')
                                 ->sum('amount');
        $productPaymentTotal = Payment::where('supplier_id', $supplier->id)
                                     ->where('type', 'product_payment')
                                     ->sum('amount');
        $supplier->updateDebt($toSupplierTotal, $productPaymentTotal);

        return redirect()->route('payments-to-supplier.index')
                        ->with('success', 'To‘lov o‘chirildi va qarz balansi yangilandi');
    }
}