<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\Payment;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class SupplierController extends Controller
{
    public function index(Request $request)
    {
        $query = Supplier::query();

        $userId = $request->input('user_id');
        if ($userId) {
            $query->where('user_id', $userId);
        }

        $debtType = $request->input('debt_type');
        if ($debtType === 'we_owe') {
            $query->where('debt_balance', '<', 0);
        } elseif ($debtType === 'they_owe') {
            $query->where('debt_balance', '>', 0);
        }

        $date = $request->input('date');
        if ($date) {
            $query->whereDate('created_at', '>=', $date);
        }

        $suppliers = $query->paginate(10);

        $balanceQuery = Supplier::query();
        if ($userId) {
            $balanceQuery->where('user_id', $userId);
        }
        if ($date) {
            $balanceQuery->whereDate('created_at', '>=', $date);
        }
        $weOweBalance = $debtType !== 'they_owe' ? $balanceQuery->where('debt_balance', '<', 0)->sum('debt_balance') : 0;
        $theyOweBalance = $debtType !== 'we_owe' ? $balanceQuery->where('debt_balance', '>', 0)->sum('debt_balance') : 0;

        $users = User::all();

        return view('suppliers.index', compact('suppliers', 'weOweBalance', 'theyOweBalance', 'users'));
    }

    public function create()
    {
        return view('suppliers.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'debt_balance' => 'required|numeric|min:0',
            'debt_type' => 'required_if:debt_balance,>,0|in:we_owe,they_owe',
        ]);

        try {
            $debtBalance = $request->debt_balance;
            if ($debtBalance > 0) {
                $debtBalance = $request->debt_type === 'we_owe' ? -$debtBalance : $debtBalance;
            }

            Supplier::create([
                'name' => $request->name,
                'phone' => $request->phone,
                'debt_balance' => $debtBalance,
                'user_id' => Auth::user()->id,
            ]);

            return redirect()->route('suppliers.index')->with('success', 'Yetkazib beruvchi muvaffaqiyatli qo‘shildi');
        } catch (\Exception $e) {
            Log::error('Yetkazib beruvchi qo‘shishda xato: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Yetkazib beruvchi qo‘shishda xato yuz berdi');
        }
    }

    public function show(Supplier $supplier)
    {
        $supplier->load('user');

        $payments = Payment::select(
            'id',
            'type',
            DB::raw('CAST(amount AS DECIMAL(15,2)) AS amount'),
            'source',
            'description',
            'images',
            'created_at',
            'is_cancelled',
            'user_id',
            DB::raw("'payments' AS source_table")
        )
        ->where('supplier_id', $supplier->id);

        $transactions = Transaction::select(
            'id',
            'type',
            DB::raw('CAST(amount AS DECIMAL(15,2)) AS amount'),
            'source',
            'description',
            'images',
            'created_at',
            DB::raw('0 AS is_cancelled'),
            'user_id',
            DB::raw("'transactions' AS source_table")
        )
        ->where('supplier_id', $supplier->id)
        ->where('type', 'product_return');

        $operations = $payments->union($transactions)
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('suppliers.show', compact('supplier', 'operations'));
    }

    public function edit(Supplier $supplier)
    {
        if (!Auth::user()->isAdmin() && Auth::user()->id !== $supplier->user_id) {
            Log::warning('Tahrirlash rad etildi: Foydalanuvchi na admin, na yetkazib beruvchi yaratuvchisi', [
                'user_id' => Auth::id(),
                'supplier_id' => $supplier->id,
            ]);
            return redirect()->route('suppliers.show', $supplier->id)->with('error', 'Sizda bu yetkazib beruvchini tahrirlash huquqi yo‘q.');
        }

        return view('suppliers.edit', compact('supplier'));
    }

    public function update(Request $request, Supplier $supplier)
    {
        if (!Auth::user()->isAdmin() && Auth::user()->id !== $supplier->user_id) {
            Log::warning('Yangilash rad etildi: Foydalanuvchi na admin, na yetkazib beruvchi yaratuvchisi', [
                'user_id' => Auth::id(),
                'supplier_id' => $supplier->id,
            ]);
            return redirect()->route('suppliers.show', $supplier->id)->with('error', 'Sizda bu yetkazib beruvchini tahrirlash huquqi yo‘q.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
        ]);

        try {
            $supplier->update([
                'name' => $request->name,
                'phone' => $request->phone,
            ]);

            Log::info('Yetkazib beruvchi yangilandi', [
                'supplier_id' => $supplier->id,
                'user_id' => Auth::id(),
                'new_name' => $request->name,
                'new_phone' => $request->phone,
            ]);

            return redirect()->route('suppliers.show', $supplier->id)->with('success', 'Yetkazib beruvchi muvaffaqiyatli yangilandi');
        } catch (\Exception $e) {
            Log::error('Yetkazib beruvchi yangilashda xato: ' . $e->getMessage(), [
                'supplier_id' => $supplier->id,
                'user_id' => Auth::id(),
            ]);
            return redirect()->back()->with('error', 'Yetkazib beruvchi yangilashda xato yuz berdi');
        }
    }

    public function destroy(Supplier $supplier)
    {
        try {
            $supplier->delete();
            return redirect()->route('suppliers.index')->with('success', 'Yetkazib beruvchi muvaffaqiyatli o‘chirildi');
        } catch (\Exception $e) {
            Log::error('Yetkazib beruvchi o‘chirishda xato: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Yetkazib beruvchi o‘chirishda xato yuz berdi');
        }
    }

    public function returnProduct(Request $request, Supplier $supplier)
    {
        $request->validate([
            'amount' => ['required', 'numeric', 'min:0.01', 'max:1000000'],
            'description' => ['nullable', 'string', 'max:1000'],
            'images.*' => ['nullable', 'image', 'mimes:jpg,jpeg,png', 'max:2048'],
            'images' => ['max:5'],
        ]);

        try {
            if (!Auth::check()) {
                Log::error('Foydalanuvchi autentifikatsiya qilinmagan');
                return redirect()->back()->with('error', 'Foydalanuvchi autentifikatsiya qilinmagan.');
            }

            $user = Auth::user();
            if (!$user) {
                Log::error('Foydalanuvchi topilmadi: user_id=' . (Auth::id() ?? 'unknown'));
                return redirect()->back()->with('error', 'Foydalanuvchi topilmadi.');
            }

            $totalProductPayments = Payment::where('supplier_id', $supplier->id)
                ->where('type', 'product_payment')
                ->where('is_cancelled', false)
                ->sum('amount');
            if ($request->amount > $totalProductPayments) {
                Log::warning('Qaytariladigan summa to‘lovlardan oshdi: kiritilgan=' . $request->amount . ', jami=' . $totalProductPayments);
                return back()->withErrors(['amount' => 'Qaytariladigan summa mahsulot to‘lovlaridan (' . number_format($totalProductPayments, 2, '.', ' ') . ' so‘m) oshib ketdi.']);
            }

            $imagePaths = [];
            if ($request->hasFile('images')) {
                if (!Storage::disk('public')->exists('returns')) {
                    Storage::disk('public')->makeDirectory('returns');
                }
                foreach ($request->file('images') as $image) {
                    $path = $image->store('returns', 'public');
                    $imagePaths[] = $path;
                }
            }

            $transaction = Transaction::create([
                'user_id' => $user->id,
                'type' => 'product_return',
                'amount' => -$request->amount,
                'description' => $request->description,
                'images' => !empty($imagePaths) ? json_encode($imagePaths) : null,
                'supplier_id' => $supplier->id,
            ]);

            $supplier->debt_balance += $request->amount;
            $supplier->save();

            Log::info('Mahsulot qaytarildi: supplier_id=' . $supplier->id . ', amount=' . $request->amount . ', transaction_id=' . $transaction->id . ', user_id=' . $user->id . ', new_debt_balance=' . $supplier->debt_balance);

            return redirect()->route('suppliers.show', $supplier->id)->with('success', 'Mahsulot muvaffaqiyatli qaytarildi.');
        } catch (\Exception $e) {
            Log::error('Mahsulot qaytarishda xato: ' . $e->getMessage(), [
                'request' => $request->all(),
                'supplier_id' => $supplier->id,
                'user_id' => Auth::id() ?? 'unknown',
            ]);
            return redirect()->back()->with('error', 'Mahsulot qaytarishda xato yuz berdi: ' . $e->getMessage());
        }
    }
}