<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Payment;
use App\Models\Supplier;
use App\Models\UserBalance;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TransactionController extends Controller
{
    public function index()
    {
        $results = collect();

        $transactions = Transaction::with('user', 'category')
            ->selectRaw("id, user_id, NULL as supplier_id, type, amount, source, description, images, is_cancelled, cancelled_at, created_at, 'transactions' as source_table")
            ->get()
            ->map(function ($item) {
                $item->user_name = $item->user ? $item->user->name : null;
                $item->supplier_name = null;
                $item->related_entity = $item->category_id ? ($item->category ? $item->category->name : null) : null;
                return $item;
            });

        $payments = Payment::with('user', 'supplier')
            ->selectRaw("id, user_id, supplier_id, type, amount, source, description, images, is_cancelled, cancelled_at, created_at, 'payments' as source_table")
            ->get()
            ->map(function ($item) {
                $item->user_name = $item->user ? $item->user->name : null;
                $item->supplier_name = $item->supplier ? $item->supplier->name : null;
                $item->related_entity = $item->supplier ? $item->supplier->name : null;
                return $item;
            });

        $results = $transactions->merge($payments)->sortByDesc('created_at')->paginate(10);

        return view('transactions.history', compact('results'));
    }

    public function cancelTransaction(Request $request, $id)
    {
        if (!Auth::user()->isAdmin()) {
            Log::warning('Bekor qilish rad etildi: Foydalanuvchi admin emas', ['user_id' => Auth::id()]);
            return redirect()->back()->with('error', 'Faqat admin amaliyotni bekor qila oladi.');
        }

        try {
            $transaction = Transaction::findOrFail($id);
            if ($transaction->is_cancelled) {
                Log::info('Amaliyot allaqachon bekor qilingan', ['transaction_id' => $id]);
                return redirect()->back()->with('error', 'Amaliyot allaqachon bekor qilingan.');
            }

            DB::transaction(function () use ($transaction) {
                $rawAmount = str_replace(',', '', $transaction->amount);
                $amount = abs((float) $rawAmount);
                if ($amount <= 0) {
                    throw new \Exception('Tranzaksiya summasi noto‘g‘ri: ' . $transaction->amount);
                }

                $transaction->update([
                    'is_cancelled' => true,
                    'cancelled_at' => now(),
                ]);

                Log::info('Amaliyot bekor qilindi', ['transaction_id' => $transaction->id]);
            });

            return redirect()->back()->with('success', 'Amaliyot muvaffaqiyatli bekor qilindi.');
        } catch (\Exception $e) {
            Log::error('Amaliyot bekor qilishda xato', [
                'transaction_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->back()->with('error', 'Amaliyot bekor qilishda xato yuz berdi');
        }
    }

    public function cancelPayment(Request $request, $id)
    {
        if (!Auth::user()->isAdmin()) {
            Log::warning('To‘lov bekor qilish rad etildi: Foydalanuvchi admin emas', ['user_id' => Auth::id()]);
            return redirect()->back()->with('error', 'Faqat admin to‘lovni bekor qila oladi.');
        }

        try {
            $payment = Payment::with('supplier')->findOrFail($id);
            if ($payment->is_cancelled) {
                Log::info('To‘lov allaqachon bekor qilingan', ['payment_id' => $id]);
                return redirect()->back()->with('error', 'To‘lov allaqachon bekor qilingan.');
            }

            $rawAmount = str_replace(',', '', $payment->amount);
            $amount = abs((float) $rawAmount);
            if ($amount <= 0) {
                throw new \Exception('To‘lov summasi noto‘g‘ri: ' . $payment->amount);
            }

            DB::transaction(function () use ($payment, $amount) {
                $supplier = $payment->supplier;
                if (!$supplier) {
                    throw new \Exception('Yetkazib beruvchi topilmadi: supplier_id=' . $payment->supplier_id);
                }

                if ($payment->type == 'product_payment') {
                    // Mahsulot to‘lovi bekor qilinganda qarz kamayadi
                    $supplier->debt_balance -= $amount;
                } elseif ($payment->type == 'payment_to_supplier') {
                    // Yetkazib beruvchiga to‘lov bekor qilinganda qarz oshadi
                    $supplier->debt_balance += $amount;

                    // Foydalanuvchi balansiga summa qaytariladi
                    $userBalance = UserBalance::where('user_id', Auth::id())->first();
                    if ($userBalance) {
                        $userBalance->balance += $amount;
                        $userBalance->save();
                        Log::info('Shaxsiy balans yangilandi', ['user_id' => Auth::id(), 'amount' => $amount]);
                    } else {
                        Log::warning('Shaxsiy balans topilmadi', ['user_id' => Auth::id()]);
                    }
                } else {
                    throw new \Exception('Noto‘g‘ri to‘lov turi: ' . $payment->type);
                }

                $supplier->save();

                $payment->update([
                    'is_cancelled' => true,
                    'cancelled_at' => now(),
                ]);

                Log::info('To‘lov bekor qilindi', [
                    'payment_id' => $payment->id,
                    'supplier_id' => $payment->supplier_id,
                    'type' => $payment->type,
                    'amount' => $amount,
                    'new_debt_balance' => $supplier->debt_balance
                ]);
            });

            return redirect()->back()->with('success', 'To‘lov muvaffaqiyatli bekor qilindi.');
        } catch (\Exception $e) {
            Log::error('To‘lov bekor qilishda xato', [
                'payment_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->back()->with('error', 'To‘lov bekor qilishda xato yuz berdi: ' . $e->getMessage());
        }
    }
}